<?php

namespace ODE_API\V8\Service;

use ODE_API\V8\BeanDecorator\BeanManager;
use ODE_API\V8\JsonApi\Helper\AttributeObjectHelper;
use ODE_API\V8\JsonApi\Helper\PaginationObjectHelper;
use ODE_API\V8\JsonApi\Helper\RelationshipObjectHelper;
use ODE_API\V8\JsonApi\Response\DossierResponse;

use ODE_API\V8\Param\DossierSetParams;

use ODE\Generateur\Factory\OdeFormFactory;

use ODE\Model\DispositifModel;
use ODE\Model\DemandeurModel;
use ODE_API\V8\JsonApi\Response\DispositifResponse;


use ODE\Helper\OdePdfHelper;
use ODE\Mailer\OdeEmail;
use ODE\Mailer\OdeMailer;

use OPS_dossier;
use BeanFactory;
use DateTime;
use Exception;
use TimeDate;
use Slim\Http\Request;


class HorsCompteService
{
    /**
     * @var BeanManager
     */
    protected $beanManager;

    /**
     * @var AttributeObjectHelper
     */
    protected $attributeHelper;

    /**
     * @var RelationshipObjectHelper
     */
    protected $relationshipHelper;

    /**
     * @var PaginationObjectHelper
     */
    protected $paginationHelper;

    /**
     * @param BeanManager $beanManager
     * @param AttributeObjectHelper $attributeHelper
     * @param RelationshipObjectHelper $relationshipHelper
     * @param PaginationObjectHelper $paginationHelper
     */
    public function __construct(BeanManager $beanManager, AttributeObjectHelper $attributeHelper, RelationshipObjectHelper $relationshipHelper, PaginationObjectHelper $paginationHelper)
    {
        $this->beanManager = $beanManager;
        $this->attributeHelper = $attributeHelper;
        $this->relationshipHelper = $relationshipHelper;
        $this->paginationHelper = $paginationHelper;
    }

    /**
     * getListDispositif
     * @param DispositifListParams $params
     * @param $path
     * @return DispositifResponse
     * @throws AccessDeniedException
     */
    public function getListDispositif(Request $request)
    {
        global $db;

        $dispositifs = DispositifModel::getListeHc();
        $tab_dispositif = array();

        foreach ($dispositifs as $key => $dispositif) {

            $lock = true; // Peux-t-on accèder au TS ? non ! 
            $saison = false; // Affichons-nous le TS ? non ! 
            $visible_portail = false; // Doit-il être afficher dans la vue Nouveau dossier ? 

            // Gestion des saisons 
            if ($dispositif['saison'] == true) {

                $dateTime = new DateTime('NOW');
                $today = $dateTime->format('Y-m-d');

                $saison_affichage_date_debut = $dispositif['saison_affichage_date_debut'];
                $saison_affichage_date_fin = $dispositif['saison_affichage_date_fin'];
                $saison_date_debut = $dispositif['saison_date_debut'];
                $saison_date_fin = $dispositif['saison_date_fin'];

                // Affichage du dispositif ? 
                if ($saison_affichage_date_debut <= $today && $today <= $saison_affichage_date_fin) {

                    $date_fin = new DateTime($saison_date_fin);
                    $date_fin = $date_fin->format('d/m/Y');

                    $date_debut = new DateTime($saison_date_debut);
                    $date_debut = $date_debut->format('d/m/Y');

                    // Dispositif ouvert ? 
                    if ($saison_date_debut <= $today && $today <= $saison_date_fin) {

                        $date_fin = new DateTime($saison_date_fin);
                        $date_fin = $date_fin->format('d/m/Y');

                        $lock = false;
                        $saison = "Ouvert jusqu'au " . $date_fin . " inclus";
                        $visible_portail = true;
                    } else {
                        $visible_portail = true;

                        // Le dispositif va bientôt ouvrir 
                        if ($saison_date_debut >= $today) {
                            $saison = "Ouvrira le " . $date_debut;
                        } else {
                            $saison = "Fermé depuis le " . $date_fin;
                        }
                    }
                }
            } else {
                $lock = false;
                $visible_portail = true;
            }

            $tab_dispositif[$dispositif['id']]['id'] = $dispositif['id'];
            $tab_dispositif[$dispositif['id']]['name'] = $dispositif['name'];
            $tab_dispositif[$dispositif['id']]['libelle_court'] = $dispositif['libelle_court'];
            $tab_dispositif[$dispositif['id']]['couleur'] = $dispositif['couleur'];
            $tab_dispositif[$dispositif['id']]['icone'] = $dispositif['icone'];
            $tab_dispositif[$dispositif['id']]['visible_usager'] = $dispositif['visible_usager'];
            $tab_dispositif[$dispositif['id']]['aide'] = $dispositif['description'];
            $tab_dispositif[$dispositif['id']]['saison'] = $saison;
            $tab_dispositif[$dispositif['id']]['lock'] = $lock;
            $tab_dispositif[$dispositif['id']]['visible_portail'] = $visible_portail;
        }

        $response = new DispositifResponse();
        $response->setData($tab_dispositif);

        return $response;
    }


    /**
     * getFormByEtape
     * @param DossierSetParams $params
     * @param $path
     * @return DossierResponse
     * @throws AccessDeniedException
     */
    public function getFormByEtape(DossierSetParams $params, Request $request)
    {
        global $db, $sugar_config;
        // On récupere les données
        $donnees = $params->getData();

        // On vérifie que "form_type" n'est pas vide 
        if (empty($donnees["form_type"])) {
            throw new Exception("Le parametre 'form_type' est obligatoire", 401);
        }

        // On vérifie que "form_type" est égal à creation ou edition ou detail
        if (!in_array($donnees["form_type"], ["creation", "edition", "detail"])) {
            throw new Exception("Le parametre 'form_type' doit etre égal à une des valeurs : 'creation', 'edition' ou 'detail' ", 401);
        }

        // On récupére le type de formulaire à retourner et on le supprime du tableau de données
        $form_type = $donnees["form_type"];
        unset($donnees["form_type"]);

        // On vérifie que l'id dossier n'est pas vide si il s'agit d'une vue de type "edition" ou "detail"
        if ($form_type !== "creation" && empty($donnees["dossier_id"])) {
            $response = new DossierResponse();
            $response->setData(["erreur" => "Les paramètres sont incorrects"]);
            return $response;
        }

        // On vérifie que le dossier existe si il s'agit d'une vue de type "edition" ou "detail"
        $obj_dossier = (!empty($donnees["dossier_id"])) ? BeanFactory::getBean('OPS_dossier', $donnees["dossier_id"]) : false;
        if ($form_type !== "creation" && empty($obj_dossier->id)) {
            // Vérification par le profil
            $response = new DossierResponse();
            $response->setData(["erreur" => "Le dossier n'existe pas"]);
            return $response;
        }

        $dispositif = $this->beanManager->getBeanSafe("OPS_dispositif", $donnees["dispositif_id"]);

        // Vérification d'accès au dispositif dans le cadre d'une édition ( vérif saison )
        if ($form_type == "creation") {
            $is_ouvert = $this->verif_saison($dispositif);

            if (!$is_ouvert) {
                // Vérification par le profil
                $response = new DossierResponse();
                $response->setData(["erreur" => "Ce dispositif est temporairement désactivé ou n'est pas disponible pour le moment."]);
                return $response;
            }
        }

        if ($form_type == "edition") {

            $dispositif = $this->beanManager->getBeanSafe("OPS_dispositif", $donnees["dispositif_id"]);

            $is_ouvert = $this->verif_saison($dispositif);

            if (!$is_ouvert && $obj_dossier->incomplet == false) {
                // Vérification par le profil
                $response = new DossierResponse();
                $response->setData(["erreur" => "Vous ne pouvez plus modifier ce brouillon car le formulaire correspondant n'est plus disponible"]);
                return $response;
            }
        }

        if (!$donnees['demandeur_id']) { // On est sur une création en hors-compte donc pas d'individu de passé en paramètre
            $cur_individu = BeanFactory::newBean('OPS_individu');
            $cur_individu->salutation = $donnees['civilite'];
            $cur_individu->last_name = $donnees['nom'];
            $cur_individu->first_name = $donnees['prenom'];
            $cur_individu->email1 = $donnees['email'];
            $cur_individu->canal = 'hors_compte';

            $donnees['demandeur_id'] = $cur_individu->save();
        }

        // Vérification que l'usager connecté a le droit de modification sur le dossier 
        if ($donnees['demandeur_id'] != $obj_dossier->ops_individu_id && $form_type !== "creation") {
            $response = new DossierResponse();
            $response->setData(["erreur" => "Vous n'avez pas accès à ce dossier"]);
            return $response;
        }

        // Vérification que le dossier soit en brouillon 
        if ($obj_dossier->brouillon == "non" && $obj_dossier->incomplet == false && $form_type != "detail") {
            $obj_statut = $this->beanManager->getBeanSafe("OPS_statut", $obj_dossier->ops_statut_id);
            if (!$obj_statut->modifiable_usager) {
                $response = new DossierResponse();
                $response->setData(["erreur" => "Le dossier est en cours de traitement par nos services."]);
                return $response;
            }
        }

        // On vérifie les données recus 
        $verification_donnees = $this->getDataFormByEtape($donnees);
        if ($verification_donnees["statut"] === "err") {
            throw new Exception('Données invalides : ' . $verification_donnees["data"], 401);
        }

        // On récupere les données formatées et on ajoute le parametre "création"
        $donnees_formated = $verification_donnees["data"];
        $donnees_formated["vue"]["creation"] = ($form_type === "creation") ? "true" : "false";

        $donnees_formated["dispositif"]["canal"] = $donnees['canal'];

        // On initilise OdeFormFactory selon le type de form
        if ($form_type === "creation") {
            // Ajout du bénéficiaire et PM 
            $donnees_formated['ops_personne_morale'] = $donnees['ops_personne_morale'];
            $donnees_formated['brouillon'] = true;
            $donnees_formated['beneficiaire_id'] = $donnees['beneficiaire_id'];

            // On récupere l'id du dossier 
            if (empty($obj_dossier->id)) {
                $dossier_id = OPS_dossier::createDossierByEtape($donnees_formated);
                $obj_dossier = BeanFactory::getBean('OPS_dossier', $dossier_id);

                // On créé le code de suivi
                $code_suivi = $this->genCodeSuivi();
                $obj_dossier->code_suivi = $code_suivi;
                $obj_dossier->save();

                // #3172 : num_dossier non récupéré - Impossible via un bean à cause de l'auto incremente
                $requete = $db->query("SELECT id, num_dossier from ops_dossier where id='" . $obj_dossier->id . "' ");
                $rowResults = array();
                while ($row = $db->fetchRow($requete)) {
                    $num_dossier = $row['num_dossier'];
                }
                $obj_dossier->num_dossier = $num_dossier;
            }
        } else if ($form_type === "edition") {
            $donnees_formated['ops_personne_morale'] = $obj_dossier->ops_personne_morale;
            $donnees_formated['brouillon'] = $obj_dossier->brouillon;
            $donnees_formated['beneficiaire_id'] = $obj_dossier->beneficiaire_id;
        } else if ($form_type === "detail") {
            $donnees_formated['ops_personne_morale'] = $obj_dossier->ops_personne_morale;
            $donnees_formated['beneficiaire_id'] = $obj_dossier->beneficiaire_id;
            $donnees_formated['brouillon'] = $obj_dossier->brouillon;
            $donnees["demandeur"]["id"] = $dossier->ops_individu_id;
            $editable = false;
        }

        $formFactory = new OdeFormFactory($obj_dossier, $editable, $donnees_formated);


        // On récupere les etapes ( On parle d'onglet coté générateur )
        $etapes = $formFactory->getEtapes();
        // Vérification des conditions
        if (isset($etapes[$donnees_formated["etape_ordre"]])) {
            $ordre = $donnees_formated["etape_ordre"];
        } else {
            $ordre = $donnees_formated["etape_ordre"];
            $fin = true;
            foreach ($etapes as $key => $etape) {
                $ordre = $ordre + 1;
                if (!isset($etapes[$ordre])) {
                    continue;
                } else {
                    $fin = false;
                    break;
                }
            }
        }
        // On réécrit le construct pour la gestion de la modification 
        if ($etapes[$ordre]['modifiable'] == false) {
            $formFactory = new OdeFormFactory($obj_dossier, false, $donnees_formated);
        }

        $etape_html = $formFactory->getEtapeHtml($etapes[$ordre]['ordre']);

        // On récupere certaines données complementaires
        $infos = $this->getDataCompltFormByEtape($obj_dossier, $etapes, $dossier_id, $donnees);


        # Complement Implémentation #2526 Onglet [response]
        if ($form_type === "detail") {

            $infos["flag_valeur_1"] = $dispositif->flag_valeur_1;
            $infos["flag_valeur_2"] = $dispositif->flag_valeur_2;
            $infos["flag_valeur_3"] = $dispositif->flag_valeur_3;
        }


        // Gestion de la fin du formulaire si conditionné 
        if ($fin === true) {

            $obj_dossier->brouillon = "non";
            $obj_dossier->save();

            // Récupération du dispositif lié au dossier pour récupérer le guide d'instruction
            $dispositifs = $obj_dossier->get_linked_beans('ops_dispositif_ops_dossier', 'OPS_dispositif');
            $dispositif = (is_array($dispositifs) && count($dispositifs) == 1) ? $dispositifs[0] : false;

            // Récupération du guide d'instruction
            $guide_instructions = ($dispositif !== false) ? $dispositif->get_linked_beans('ops_guide_instruction_ops_dispositif', 'OPS_guide_instruction') : array();
            $guide_instruction = (is_array($guide_instructions) && count($guide_instructions) == 1) ? $guide_instructions[0] : false;

            // Gestion d'un dossier incomplet  && existe un statut de complétude mise à jour du dossier 
            if ($obj_dossier->incomplet == true && !empty($guide_instruction->ops_statut_completude_id)) {

                $template = $sugar_config['opensocle']['notif_usager_dossier_modification'];
                if (isset($template) && !empty($template)) {

                    $obj_individu = $this->beanManager->getBeanSafe("OPS_individu", $obj_dossier->ops_individu_id);

                    # On initialise le mailer 
                    $mailer = new OdeMailer();

                    # On initialise l'email 
                    $ode_email = new OdeEmail([
                        'bean_source_id' => $obj_dossier->id,
                        'bean_source_name' => 'OPS_dossier',
                        'bean_historisation_id' => $obj_dossier->ops_individu_id,
                        'bean_historisation_name' => 'OPS_individu',
                        'email_template_id' => $template,
                        'dest_to' => $obj_individu->email1,
                    ]);

                    # On déclenche l'envoie
                    $mailer->send($ode_email);
                } else {
                    $GLOBALS['log']->fatal("OPS_dossier - sendNotification : Pas te template mail associé à la création d'un dossier");
                }


                $obj_dossier->updateStatut($guide_instruction->ops_statut_completude_id);
                $obj_dossier->save();

                // Notification dans la cloche à l'agent 
                $user_id = $obj_dossier->assigned_user_id;
                $obj_user = BeanFactory::getBean('Users', $user_id);

                $alert = BeanFactory::newBean('Alerts');
                $alert->name = 'Dossier complété';
                $alert->description = 'Dossier numéro :' . $obj_dossier->num_dossier;
                $alert->url_redirect = 'index.php?module=OPS_dossier&action=DetailView&record=' . $obj_dossier->id;
                $alert->assigned_user_id = $obj_user->id;
                $alert->type = 'info';
                $alert->is_read = 0;
                $alert->save();
            } elseif (empty($obj_dossier->ops_statut_id)) {

                // Initialisation du statut et de l'étape du dossier
                $obj_dossier = OPS_dossier::initStatutEtapeDossier($obj_dossier, $dispositif->id);
                $obj_dossier->save();

                $template = $sugar_config['opensocle']['notif_usager_dossier_hc_creation'];
                if (isset($template) && !empty($template)) {
                    $obj_individu = $this->beanManager->getBeanSafe("OPS_individu", $obj_dossier->ops_individu_id);

                    # On initialise le mailer 
                    $mailer = new OdeMailer();

                    # On initialise l'email 
                    $ode_email = new OdeEmail([
                        'bean_source_id' => $obj_dossier->id,
                        'bean_source_name' => 'OPS_dossier',
                        'bean_historisation_id' => $obj_dossier->ops_individu_id,
                        'bean_historisation_name' => 'OPS_individu',
                        'email_template_id' => $template,
                        'dest_to' => $obj_individu->email1,
                    ]);

                    # On déclenche l'envoie
                    $mailer->send($ode_email);
                } else {
                    $GLOBALS['log']->fatal("OPS_dossier - sendNotification : Pas te template mail associé à la création d'un dossier");
                }
            }
        }

        $response = new DossierResponse();
        $response->setData(["infos" => $infos, "etapes" => $etapes, "etape_html" => base64_encode($etape_html), "fin" => $fin]);

        return $response;
    }

    /**
     * Fonction créant un code de validation sur le dossier et l'envoyant par mail
     *
     * @param DossierSetParams $params
     * @param Request $request
     * @return boolean $envoi_mail
     */
    public function sendCodeValidationHc(DossierSetParams $params, Request $request)
    {
        global $sugar_config, $timedate;
        // On récupere les données
        $donnees = $params->getData();

        $dossier_individu = $this->getDossierHc($donnees);
        if (!empty($dossier_individu)) {
            $code = $this->genCodeValidation();

            $dossier = BeanFactory::getBean('OPS_dossier', $dossier_individu['dossier']['id_dossier']);
            $dossier->code_validation = $code;
            if (isset($sugar_config['ops_hors_compte']['temps_expiration_code_validation']) && !empty($sugar_config['ops_hors_compte']['temps_expiration_code_validation'])) {
                $minutes = $sugar_config['ops_hors_compte']['temps_expiration_code_validation'];
            } else {
                $minutes = 5;
            }

            $date = $timedate->asDb($timedate->getNow()->get("+" . $minutes . " minutes"));
            $dossier->date_expiration_code_validation = $date;
            $dossier->save();

            if ($sugar_config['ops_hors_compte']['modele_mail_validation']) {
                # On initialise le mailer 
                $mailer = new OdeMailer();

                # On initialise l'email 
                $ode_email = new OdeEmail([
                    'bean_source_id' => $dossier_individu['dossier']['id_dossier'],
                    'bean_source_name' => 'OPS_dossier',
                    'bean_historisation_id' => $dossier_individu['individu']['id_individu'],
                    'bean_historisation_name' => 'OPS_individu',
                    'email_template_id' => $sugar_config['ops_hors_compte']['modele_mail_validation'],
                    'dest_to' => $dossier_individu['individu']['email_individu'],
                ]);

                # On déclenche l'envoie
                $envoi_mail = $mailer->send($ode_email);
            }

            return ["mail_envoye" => $envoi_mail];
        } else {
            throw new Exception("Aucun dossier n'a été trouvé. Veuillez corriger votre recherche.", 401);
        }

        return;
    }

    /**
     * Fonction de vérification de validité du code de validation
     *
     * @param DossierSetParams $params
     * @param Request $request
     * @return void
     */
    public function verifCodeValidationHc(DossierSetParams $params, Request $request)
    {
        global $timedate;
        // On récupere les données
        $donnees = $params->getData();

        $dossier_individu = $this->getDossierHc($donnees);
        if (!empty($dossier_individu)) {
            $dossier = BeanFactory::getBean('OPS_dossier', $dossier_individu['dossier']['id_dossier']);

            $date = DateTime::createFromFormat($timedate->get_date_time_format(), $dossier->date_expiration_code_validation);
            if (
                $dossier->code_validation == $donnees['code_validation']
                && $timedate->asDb($date) >= $timedate->nowDb()
            ) {
                $response = ["code_valide" => "1"];
            } else {
                throw new Exception('Le code de validation est incorrect ou a expiré.', 401);
            }

            return $response;
        } else {
            throw new Exception("Aucun dossier n'a été trouvé. Veuillez corriger votre recherche.", 401);
        }

        return;
    }

    /**
     * Fonction qui permet de récupérer le dossier via le combo "code_suivi" et "email1" de l'individu
     *
     * @param array $donnees le tableau des informations saisies par l'usager
     * @return array On retourne un tableau contenant les informations du dossier et de l'individu
     */
    private function getDossierHc($donnees)
    {
        $bean_dossier = BeanFactory::newBean('OPS_dossier');
        $dossier = $bean_dossier->retrieve_by_string_fields(
            array('code_suivi' => $donnees['code_suivi'])
        );

        // Si un dossier comportant le code de suivi saisi par l'usager existe, on retourne va vérifier la casse des caractères
        if (!empty($dossier)) {
            if ($dossier->code_suivi === $donnees['code_suivi']) {
            } else {
                throw new Exception("Aucun dossier n'a été trouvé. Veuillez corriger votre recherche.", 401);
            }
        } else { // sinon, on retourne une erreur
            throw new Exception("Aucun dossier n'a été trouvé. Veuillez corriger votre recherche.", 401);
        }

        $dossier->load_relationship('ops_individu_ops_dossier');
        $individu_ids = $dossier->ops_individu_ops_dossier->get();
        $individu = BeanFactory::getBean('OPS_individu', $individu_ids[0]);

        // Si l'adresse mail de l'individu rattaché au dossier ne correspond pas à la saisie de l'usager, on retourne une erreur
        if ($individu->email1 != $donnees['email']) {
            throw new Exception("Aucun dossier n'a été trouvé. Veuillez corriger votre recherche.", 401);
        }

        return [
            "dossier" => array("id_dossier" => $dossier->id),
            "individu" => array("id_individu" => $individu->id, "email_individu" => $individu->email1)
        ];
    }

    /**
     * updateDossierByEtape
     * @param DossierSetParams $params
     * @param $path
     * @return DossierResponse
     * @throws AccessDeniedException
     */
    public function updateDossierByEtape(DossierSetParams $params, Request $request)
    {
        global $sugar_config;
        // On récupere les données
        $donnees = $params->getData();
        $donneesFormated = $this->getDataFormByEtape($donnees);

        // On vérifie que les données ne sont pas vide 
        if (!is_array($donnees) || count($donnees) === 0) {
            throw new Exception('Données vide', 401);
        }

        // On vérifie que l'id du demandeur n'est pas vide
        if (empty($donnees["demandeur_id"])) {
            throw new Exception("L'id du demandeur est obligatoire", 401);
        } else {
            $donnees["demandeur"] = $donnees["demandeur_id"];
            unset($donnees["demandeur_id"]);
        }

        // On vérifie que l'id du dispositif n'est pas vide
        if (empty($donnees["dispositif_id"])) {
            throw new Exception("L'id du dispositif est obligatoire", 401);
        } else {
            $donnees["dispositif"] = $donnees["dispositif_id"];
            unset($donnees["dispositif_id"]);
        }

        // On vérifie que canal n'est pas vide
        if (empty($donnees["canal"]) && !isset($donnees["canal"])) {
            $donnees["canal"] = 'guichet';
        }

        // On vérifie que l'id du dossier a modifié n'est pas vide
        if (empty($donnees["dossier_id"])) {
            throw new Exception("L'id du dossier est obligatoire", 401);
        } else {
            $dossier_id = $donnees["dossier_id"];
            unset($donnees["dossier_id"]);
        }

        $brouillon = $donnees["save_brouillon"];
        unset($donnees["save_brouillon"]);

        // On récupere l'etape du dossier et on le supprime des données du formulaire
        if (isset($donnees["etape_ordre"])) {
            $etape_ordre = $donnees["etape_ordre"];
            unset($donnees["etape_ordre"]);
        } else {
            $etape_ordre = "1";
        }

        $obj_dossier = $this->beanManager->getBeanSafe("OPS_dossier", $dossier_id);
        $formFactory = new OdeFormFactory($obj_dossier, false, $donneesFormated['data']);
        $etapes = $formFactory->getEtapes();

        // Si l'id du dossier est vide => On crée le dossier sinon on le met à jour
        $dossier_id = OPS_dossier::upDateDossierByEtape($dossier_id, $donnees, $etapes[$etape_ordre]['ordre'], $brouillon);

        $obj_dossier = $this->beanManager->getBeanSafe("OPS_dossier", $dossier_id);

        // Gestion d'un dossier incomplet 
        if ($donnees['form_type'] == "completude" && $donnees['terminer'] == true && $obj_dossier->incomplet == true) {

            // Récupération du dispositif lié au dossier pour récupérer le guide d'instruction
            $dispositifs = $obj_dossier->get_linked_beans('ops_dispositif_ops_dossier', 'OPS_dispositif');
            $dispositif = (is_array($dispositifs) && count($dispositifs) == 1) ? $dispositifs[0] : false;

            // Récupération du guide d'instruction
            $guide_instructions = ($dispositif !== false) ? $dispositif->get_linked_beans('ops_guide_instruction_ops_dispositif', 'OPS_guide_instruction') : array();
            $guide_instruction = (is_array($guide_instructions) && count($guide_instructions) == 1) ? $guide_instructions[0] : false;

            // S'il existe un statut de complétude mise à jour du dossier 
            if (!empty($guide_instruction->ops_statut_completude_id)) {
                $obj_dossier->updateStatut($guide_instruction->ops_statut_completude_id);

                // Notification dans la cloche à l'agent 
                $user_id = $obj_dossier->assigned_user_id;
                $obj_user = BeanFactory::getBean('Users', $user_id);

                $alert = BeanFactory::newBean('Alerts');
                $alert->name = 'Dossier complété';
                $alert->description = 'Dossier numéro :' . $obj_dossier->num_dossier;
                $alert->url_redirect = 'index.php?module=OPS_dossier&action=DetailView&record=' . $obj_dossier->id;
                $alert->assigned_user_id = $obj_user->id;
                $alert->type = 'info';
                $alert->is_read = 0;
                $alert->save();
            }

            $template = $sugar_config['opensocle']['notif_usager_dossier_modification'];

            if (isset($template) && !empty($template)) {
                $obj_individu = $this->beanManager->getBeanSafe("OPS_individu", $obj_dossier->ops_individu_id);

                # On initialise le mailer 
                $mailer = new OdeMailer();

                # On initialise l'email 
                $ode_email = new OdeEmail([
                    'bean_source_id' => $obj_dossier->id,
                    'bean_source_name' => 'OPS_dossier',
                    'bean_historisation_id' => $obj_dossier->ops_individu_id,
                    'bean_historisation_name' => 'OPS_individu',
                    'email_template_id' => $template,
                    'dest_to' => $obj_individu->email1,
                ]);

                # On déclenche l'envoie
                $mailer->send($ode_email);
            } else {
                $GLOBALS['log']->fatal("DossiersService - sendNotification : Pas te template mail associé à la complétude d'un dossier");
            }
        }


        // Si l'id est vide on retourne une erreur
        if (empty($dossier_id)) {
            throw new Exception('Erreur de création du dossier', 401);
        }

        // On retourne l'id du dossier
        $response = new DossierResponse();
        $response->setData(["dossier_id" => $dossier_id]);

        return $response;
    }

    private function verif_saison($dispositif)
    {

        $retour = true;

        // Gestion des saisons 
        if ($dispositif->saison == true) {

            $dateTime = new DateTime('NOW');
            $today = $dateTime->format('Y-m-d');

            $saison_date_debut = $dispositif->saison_date_debut;
            $saison_date_fin = $dispositif->saison_date_fin;

            $date = date_create_from_format('d/m/Y', $saison_date_debut);
            $saison_date_debut = date_format($date, 'Y-m-d');


            $date_fin = date_create_from_format('d/m/Y', $saison_date_fin);
            $saison_date_fin = date_format($date_fin, 'Y-m-d');

            $retour = false;

            // Dispositif ouvert ? 
            if ($saison_date_debut <= $today && $today <= $saison_date_fin) {
                $retour = true;
            }
        }

        return $retour;
    }

    private function getDataFormByEtape($donnees)
    {
        $donnees_formated = [];
        $libelle_erreur = "";

        do {
            if (empty($donnees["demandeur_id"])) {
                $libelle_erreur = "L'id du demandeur est vide";
                break;
            } else {
                $donnees_formated["demandeur"]["id"] = $donnees["demandeur_id"];
            }

            if (empty($donnees["demandeur_type"])) {
                $libelle_erreur = "Le type de demandeur est vide";
                break;
            } else {
                $donnees_formated["demandeur"]["type"] = $donnees["demandeur_type"];
            }

            if (empty($donnees["dispositif_id"])) {
                $libelle_erreur = "L'id du dispositif est vide";
                break;
            }

            // On récupere l'objet dispositif
            $obj_dispositif = BeanFactory::getBean('OPS_dispositif', $donnees["dispositif_id"]);
            if (empty($obj_dispositif->id)) {
                $libelle_erreur = "Le dispositif n'a pas pu etre récupéré";
                break;
            } else {
                $donnees_formated["dispositif"] = ["id" => $obj_dispositif->id, "name" => $obj_dispositif->name];
            }

            // On récupere la vue agent du formulaire associé au dispositif
            $liste_formulaire = $obj_dispositif->get_linked_beans('ops_generateur_formulaire_ops_dispositif', 'OPS_generateur_formulaire');
            $obj_formulaire = (is_array($liste_formulaire) && count($liste_formulaire) > 0) ? $liste_formulaire[0] : false;
            $usager_vue_id = ($obj_formulaire !== false && !empty($obj_formulaire->usager_vue_id)) ? $obj_formulaire->usager_vue_id : "";
            $obj_vue = BeanFactory::getBean('OPS_generateur_vue', $usager_vue_id);
            if (empty($obj_vue->id)) {
                $libelle_erreur = "La vue usager est introuvable";
                break;
            } else {
                $donnees_formated["vue"] = ["id" => $obj_vue->id, "type" => "usager"];
            }

            $donnees_formated["etape_ordre"] = (empty($donnees["etape_ordre"])) ? 1 : $donnees["etape_ordre"];
        } while (0);

        return (empty($libelle_erreur)) ? ["statut" => "ok", "data" => $donnees_formated] : ["statut" => "err", "data" => $libelle_erreur];
    }

    private function getDataCompltFormByEtape($obj_dossier, $etapes, $dossier_id, $donnees)
    {
        $obj_dispositif = BeanFactory::getBean('OPS_dispositif', $donnees["dispositif_id"]);
        // Vérification si le dossier est modifiable soit par la complétude soit par le statut
        $modifiable = $obj_dossier->incomplet;

        if (!empty($obj_dossier->ops_statut_id) && !$modifiable) {
            $obj_statut = $this->beanManager->getBeanSafe("OPS_statut", $obj_dossier->ops_statut_id, array(), 0);

            $modifiable = $obj_statut->modifiable_usager;
        }

        $tab = [
            'dossier_id' => $obj_dossier->id,
            'dossier_num' => $obj_dossier->num_dossier,
            'demandeur_id' => $obj_dossier->ops_individu_id,
            'demandeur_nom' => !empty($obj_dossier->ops_individu_id) ? DemandeurModel::getName($obj_dossier->ops_individu_id, "OPS_individu") : "",
            'ops_personne_morale' => $obj_dossier->ops_personne_morale,
            'beneficiaire_id' => $obj_dossier->beneficiaire_id,
            'demandeur_type' => $obj_dossier->type_tiers,
            'dispositif_id' => $donnees["dispositif_id"],
            'dispositif_nom' => $obj_dispositif->name,
            'etape_nb' => count($etapes),
            'etape_numero' => $donnees["etape_ordre"],
            'modifiable' => $modifiable,
            'brouillon' => $obj_dossier->brouillon,
            'etape_description' => (is_array($etapes[$donnees["etape_ordre"]]) && !empty($etapes[$donnees["etape_ordre"]]["description"])) ? $etapes[$donnees["etape_ordre"]]["description"] : "",
            'code_suivi' => $obj_dossier->code_suivi,
        ];

        $tab['profil_name'] = !empty($donnees["ops_personne_morale"]) ? DemandeurModel::getName($donnees["ops_personne_morale"], "OPS_personne_morale") : "";
        $tab['beneficiaire_name'] = !empty($donnees["beneficiaire_id"]) ? DemandeurModel::getName($donnees["beneficiaire_id"], "OPS_individu") : "";

        return $tab;
    }

    /**
     * Fonction de génération d'un code de validation pour le dossier hors-compte
     *
     * @return string code Le code de validation généré
     */
    private function genCodeValidation()
    {
        global $sugar_config;

        $chars = "0123456789";

        // On initialise le générateur de nombres aléatoires
        srand((float) microtime() * 1000000);
        $code = '';

        // Si la longueur du code n'est pas paramétrée, on met par défaut à 6 pour éviter un code vide
        $longueur_code = $sugar_config['ops_hors_compte']['longueur_code_validation'] ? $sugar_config['ops_hors_compte']['longueur_code_validation'] : '6';

        for ($i = 0; $i < $longueur_code; $i++) {
            $num = rand() % (strlen($chars) - 1);
            $tmp = substr($chars, $num, 1);
            $code = $code . $tmp;
        }

        return $code;
    }

    /**
     * Fonction de génération d'un code de suivi pour la dossier hors-compte
     *
     * @return string code Le code de suivi généré
     */
    private function genCodeSuivi()
    {
        global $sugar_config;

        $chars = "";

        // On vérifie les options activées pour le code de suivi (Admin)
        $chars .= $sugar_config['ops_hors_compte']['code_chiffres'] == true ? "0123456789" : "";
        $chars .= $sugar_config['ops_hors_compte']['code_minuscules'] == true ? "abcdefghijklmnopqrstuvwxyz" : "";
        $chars .= $sugar_config['ops_hors_compte']['code_majuscules'] == true ? "ABCDEFGHIJKLMNOPQRSTUVWXYZ" : "";
        $chars .= $sugar_config['ops_hors_compte']['special_char'] == true ? "!@#$%^&*()_-=+;:,." : "";

        // Si on a demandé à supprimer les caractères ambigus, on les supprime des caractères possibles de la chaine (Accessibilité)
        if ($sugar_config['ops_hors_compte']['eviter_char_ambigus']) {
            // Liste des caractères ambigus
            $ambig = ["l", "1", "I", "o", "O", "0"];

            $chars = str_replace($ambig, '', $chars);
        }


        // On initialise le générateur de nombres aléatoires
        srand((float) microtime() * 1000000);
        $code = '';

        // Si la longueur du code n'est pas paramétrée, on met par défaut à 6 pour éviter un code vide
        $longueur_code = $sugar_config['ops_hors_compte']['longueur_code'] ? $sugar_config['ops_hors_compte']['longueur_code'] : '6';

        for ($i = 0; $i < $longueur_code; $i++) {
            $num = rand() % (strlen($chars) - 1);
            $tmp = substr($chars, $num, 1);
            $code = $code . $tmp;
        }

        // On vérifie l'unicité du code généré (uniquement sur les dossiers en deleted "false")
        $bean_dossier = BeanFactory::newBean('OPS_dossier');
        $dossier = $bean_dossier->retrieve_by_string_fields(
            array('code_suivi' => $code)
        );
        if ($dossier->id) { // Si le code de suivi existe déjà sur un dossier, on en génère un nouveau
            $code = $this->genCodeSuivi();
        }

        return $code;
    }

    /**
     * Fonction de récupération des informations nécéssaires concernant le dossier hors-compte
     *
     * @param DossierSetParams $params
     * @param Request $request
     * @return array $tab_infos la liste des informations nécessaires pour le dossier
     */
    public function getInfosDossierHc(DossierSetParams $params, Request $request)
    {
        // On récupere les données
        $donnees = $params->getData();

        $dossier_individu = $this->getDossierHc($donnees);
        if (!empty($dossier_individu)) {

            $tab_infos = $dossier_individu;

            $dossier = BeanFactory::getBean('OPS_dossier', $dossier_individu['dossier']['id_dossier']);
            $tab_infos['dispositif']['id_dispositif'] = $dossier->ops_dispositif_id;

            $individu = BeanFactory::getBean('OPS_individu', $dossier_individu['individu']['id_individu']);
            $tab_infos['individu']['civilite'] = $individu->salutation;
            $tab_infos['individu']['nom'] = $individu->last_name;
            $tab_infos['individu']['prenom'] = $individu->first_name;
            $tab_infos['individu']['email'] = $individu->email1;

            return $tab_infos;
        } else {
            throw new Exception("Aucun dossier n'a été trouvé. Veuillez corriger votre recherche.", 401);
        }

        return;
    }
}