<?php

namespace ODE\Helper;

use TCPDF;
use BeanFactory;
use ODE\Model\DossierModel;
use ODE\Model\DispositifModel;
use ODE\Model\GenerateurFormulaireModel;
use ODE\Model\GenerateurVueModel;
use ODE\Generateur\Factory\OdeFieldFactory;

use Datetime;
use DateTimeZone;
use ODE\Helper\OdeDateHelper;

class OdePdfHelper extends TCPDF
{

    private $config;
    private $header;
    private $type_retour;
    private $dossier_id;
    private $config_path;
    private $data_retour;

    /**
     * Constructeur de la classe OdePDF
     *
     * @access public
     * @name __construct
     * @return void
     */
    public function __construct($dossier_id = null, $type_vue = null, $type_retour = null)
    {
        $this->config_path = "custom/include/Ode/Config/config_pdf.php";
        $this->config = $this->getConfig();
        $this->header = $this->getDossierData($dossier_id);
        $this->type_vue = $type_vue;
        $this->type_retour = $type_retour;
        parent::__construct(PDF_PAGE_ORIENTATION, 'mm', 'A4', true, 'UTF-8', false);
    }

    /**
     * @access private
     * @name getConfig()
     * Fonction qui retourne la configuration du pdf dynamique
     *
     *  @return array          $config: la configuration du pdf dynamique
     */
    private function getConfig()
    {
        if (file_exists($this->config_path)) {
            include $this->config_path;
            if (is_array($config) && count($config) > 0) {
                foreach (["titre", "onglet", 'section', "label", "champ"] as $param_title) {
                    $style = 'font-family:' . $config["police_pdf"] . ';';
                    foreach ($config[$param_title] as $parametre => $valeur) {
                        switch ($parametre) {
                            case 'couleur_pdf':
                                $style .= 'color:' . $valeur . ';';
                                break;
                            case 'taille_pdf':
                                $style .= 'font-size:' . $valeur . 'px;';
                                break;
                            case 'pdf_gras':
                                if ($valeur == 1) {
                                    $style .= 'font-weight: bold;';
                                }
                                break;
                            case 'pdf_italique':
                                if ($valeur == 1) {
                                    $style .= 'font-style: italic;';
                                }
                                break;
                            case 'pdf_souligner':
                                if ($valeur == 1) {
                                    $style .= 'text-decoration: underline;';
                                }
                                break;
                        }
                    }
                    $config[$param_title] = $style;
                }

                return $config;
            }
        }
        return false;
    }

    /**
     * @access private
     * @name getDossierData()
     * Fonction qui retourne les champs du dossier ( champs du formulaire associé au dispositif )
     *
     *  @param string            $dossier_id: L'id du dossier
     *  @return array            $header_data: Les données d'initilisation du Header
     */
    private function getDossierData($dossier_id)
    {
        global $current_user;

        $dossierModel = new DossierModel($dossier_id);
        $dossier = $dossierModel->getBean();

        $this->dossier_id = (is_array($dossier) && count($dossier) > 0) ? $dossier_id : false;

        $dossier_num = ($this->dossier_id !== false && !empty($dossier['num_dossier'])) ? $dossier['num_dossier'] : "";
        $dossier_name = ($this->dossier_id !== false && !empty($dossier['name'])) ? $dossier['name'] : "";
        $dossier_date = ($this->dossier_id !== false && !empty($dossier['date_entered'])) ? "Transmis le : " . $dossier['date_entered'] : "";

        $dossier_date = "";
        // On met l'heure en adéquation avec le fuseau horaire de l'utilisateur
        if($this->dossier_id !== false && !empty($dossier['date_entered']) )
        {
            $date = $dossier['date_entered'];
            $date = substr( $date , 0, 16);

            if( strpos($date , ' ') === false ){
                $format_convertion =  'Y-m-d';
                $format_sortie =  'd/m/Y';
                if( strpos( $date, '-' ) === false  ){
                    $format = 'd/m/Y';
                }
                else{
                    $format = 'Y-m-d';
                }
            }
            else{
                $format_convertion =  'Y-m-d H:i';
                $format_sortie =  'd/m/Y H:i';
                if( strpos( $date, '-' ) === false  ){
                    $format = 'd/m/Y H:i';
                }
                else{
                    $format = 'Y-m-d H:i';
                }
            }

            $date_formate = DateTime::createFromFormat( $format, $date );
            $date_formate = $date_formate->format($format_convertion );

            $UserPrefGmtOffset = $current_user->getUserDateTimePreferences()['userGmtOffset'];
            if( $UserPrefGmtOffset > 0 ){
                $dossier_date = date($format, (strtotime($date_formate) + abs($UserPrefGmtOffset*60) ) );
            }else{
                $dossier_date = date($format, (strtotime($date_formate) - abs($UserPrefGmtOffset*60) ) );
            }

            $dossier_date = "Transmis le : " . $dossier_date;
        }

        $dispositif_name = ($this->dossier_id !== false) ? $dossierModel->getDispositifName() : "";
        $demandeur_name = ($this->dossier_id !== false) ? $dossierModel->getDemandeurName() : "";

        $dispositif_name = normalizer_normalize($dispositif_name);

        return [
            "dossier_num" => $dossier_num,
            "dossier_name" => $dossier_name,
            "dossier_date" => $dossier_date,
            "dispositif_name" => $dispositif_name,
            "demandeur_name" => $demandeur_name
        ];
    }

    /**
     * @access public
     * @name generer()
     * Fonction qui genere le document PDF
     *
     */
    public function generer()
    {

        $this->data_retour = ["statut" => false, "data" => [], "erreur" => "Erreur Serveur"];

        do {

            ob_start();

            // On vérifie qu'on a bien récupérer les parametres de configuration
            if ($this->config === false) {
                $GLOBALS['log']->fatal(" OdePdfHelper :: generer => Erreur de récupération de la config du PDF dynamique ");
                break;
            }

            // On vérifie que l'id du dossier n'est pas null ou empty
            if ($this->dossier_id === false) {
                $GLOBALS['log']->fatal(" OdePdfHelper :: generer => Le dossier à télécharger est introuvable ");
                break;
            }

            // On récupére les champs du dossier avec leurs valeurs ( champs du formulaire associé au dispositif )
            $onglets = $this->getOngletDossier();
            if (!is_array($onglets) || count($onglets) === 0) {
                $GLOBALS['log']->fatal(" OdePDF :: generer => Les onglets du formulaire n'ont pas pu etre récupérer ");
                break;
            }

            // Initialiser les parametres du PDF
            $this->setPdfParams();

            // On construit le corps du PDF
            $this->setPdfCorps($onglets);

            // On sort le format selon le type_retour
            $location = (empty($_SERVER['DOCUMENT_ROOT']) || $_SERVER['DOCUMENT_ROOT'] == '/')?getcwd():$_SERVER['DOCUMENT_ROOT'];
            $file_location = $location . '/upload/dossier_' . $this->header['dossier_num'] . '.pdf';

            switch ($this->type_retour) {
                case 'base64':
                    $this->Output($file_location, 'F');
                    $file = base64_encode(file_get_contents($file_location));
                    $this->data_retour = ["statut" => true, "data" => ["name" => 'dossier_' . $this->header['dossier_num'] . '.pdf', "base64" => $file, "size" => filesize($file_location)], "erreur" => ""];
                    unlink($file_location);
                    break;
                case 'download':
                    $file = $this->Output('dossier_' . $this->header['dossier_num'] . '.pdf', 'D');
                    $this->data_retour = ["statut" => true, "data" => ["file" => $file], "erreur" => ""];
                    break;
                case 'mail':
                    $this->Output($file_location, 'F');
                    $this->data_retour = [
                        "statut" => true,
                        "data" => [
                            "name" => 'dossier_' . $this->header['dossier_num'] . '.pdf',
                            "path" => $file_location
                        ],
                        "erreur" => ""
                    ];
                    break;
                case 'file':
                    $file = $this->Output($file_location, 'I');
                    $this->data_retour = ["statut" => true, "data" => ["file" => $file], "erreur" => ""];
                    break;
                default:
                    $file = $this->Output('', 'S');
                    break;
            }
        } while (0);

        return $this->data_retour;
    }



    /**
     * @access private
     * @name getOngletDossier()
     * Fonction qui retourne les champs du dossier ( champs du formulaire associé au dispositif )
     *
     *  @return array            $champs_dossier: champs du formulaire associé au dispositif
     */
    private function getOngletDossier()
    {
        $onglets = [];

        do {

            $dossierModel = new DossierModel($this->dossier_id);
            $dispositif_id = $dossierModel->getDispositifId();
            if (empty($dispositif_id)) {
                $GLOBALS['log']->fatal(" OdePdfHelper :: getChampsDossier => Le dispositif n'a pas pu etre récupéré. ");
                break;
            }

            $dispositifModel = new DispositifModel($dispositif_id);
            $dispositif = $dispositifModel->getBean();
            if (!is_array($dispositif) || count($dispositif) === 0) {
                $GLOBALS['log']->fatal(" OdePdfHelper :: getChampsDossier => Le dispositif id = " . $dispositif_id . " est introuvable");
                break;
            }

            // On récupere la vue agent du formulaire associé au dispositif
            $formulaire_id = $dispositifModel->getFormulaireId();
            $formulaireModel = new GenerateurFormulaireModel($formulaire_id);
            $onglets['vue_id'] = ($this->type_vue === "agent") ? $formulaireModel->getVueAgentId() : $formulaireModel->getVueUsagerId();
            if (empty($onglets['vue_id'])) {
                $GLOBALS['log']->fatal(" OdePdfHelper :: getChampsDossier => La vue '" . $this->type_vue . "' id='" . $onglets['vue_id'] . "' associé formulaire id = " . $formulaire_id . " est introuvable");
                break;
            }

            $vueModel = new GenerateurVueModel($onglets['vue_id']);
            $vue = $vueModel->getBean();
            if (!is_array($vue) || count($vue) === 0) {
                $GLOBALS['log']->fatal(" OdePdfHelper :: getChampsDossier => La id = " . $onglets['vue_id'] . " est introuvable");
                break;
            }

            // On récupere les onglets de la vue 
            $onglets['onglets'] = $vueModel->getOnglets();
        } while (0);

        return $onglets;
    }


    /**
     * @access private
     * @name _initParamsPDF()
     * Fonction qui initialise les parametres du PDF
     *
     */
    private function setPdfParams()
    {
        $margin_header = 5;
        $margin_footer = 10;
        $interligne = 1.05;
        $titre =  mb_strtoupper($this->header['dossier_name'], 'UTF-8');

        // set document information
        $this->SetCreator($this->config["createur"]);
        $this->SetAuthor($this->config["createur"]);
        $this->SetTitle(str_replace("&#039;", "'", $titre));

        // set header and footer fonts
        $this->setHeaderFont(array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));
        $this->setFooterFont(array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));
        // set default monospaced font
        $this->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);

        //set margins
        $this->SetMargins($this->config["margin"]["left"], $this->config["margin"]["top"], $this->config["margin"]["right"]);
        $this->SetHeaderMargin($margin_header);
        $this->SetFooterMargin($margin_footer);

        //set auto page breaks
        $this->SetAutoPageBreak(TRUE, $this->config["margin"]["bottom"]);

        //set image scale factor
        $this->setImageScale(PDF_IMAGE_SCALE_RATIO);
        //set some language-dependent strings
        //$pdf->setLanguageArray($l);
        // set font
        $this->SetFont($this->config["police_pdf"], '', 11);

        //définiton de l'interligne
        $this->setCellHeightRatio($interligne);
        //Ajout d'une page
        $this->AddPage();
    }


    /**
     * @access private
     * @name setPdfCorps()
     * Fonction qui retourne les champs du dossier ( champs du formulaire associé au dispositif )
     *
     *  @return array            $champs_dossier: champs du formulaire associé au dispositif
     */
    private function setPdfCorps($onglets)
    {

        $corps_html = "";
        $typeProfil = '';
        $checkCondition = true;
        $obj_dossier = BeanFactory::getBean("OPS_dossier", $this->dossier_id);
        $fieldFactory = new OdeFieldFactory($obj_dossier, false, []);

        if(!empty($obj_dossier->ops_personne_morale)){
            $profilObj = BeanFactory::getBean('OPS_personne_morale',$obj_dossier->ops_personne_morale);
            $typeProfil = $profilObj->ops_type_personne_id;
        }

        $vueModel = new GenerateurVueModel($onglets['vue_id']);
        $liste_champs = $vueModel->getChamps();

        foreach ($onglets['onglets'] as $onglet) {
            
            // Vérification des conditions de l'onglet
            $conditions = json_decode( base64_decode($onglet['conditions']) ) ;
            $checkCondition = $this->checkCondition($conditions,$typeProfil);

            if( !empty($conditions->conditions)){
                foreach ($conditions->conditions as $key => $condition) {
                    $condition_remplie = false ;
                    $valeur_champ = $fieldFactory->getValueChamp($condition->champ, $liste_champs[$condition->champ]['description']) ;
                    $condition_remplie =  $this->is_condition(  $condition->value, $condition->operator ,$valeur_champ ,  $liste_champs[$condition->champ] );
                    if( $conditions->verified == "all" && $condition_remplie == false ){ // Si toutes les conditions doivent être remplies 
                        break;
                    }
                    elseif( $conditions->verified != "all" && $condition_remplie == true ){ // Si une des conditions est remplie on continue
                        break;
                    }
                }
            }

            if(!empty($typeProfil) && !empty($conditions->type_profil) && empty($checkCondition) ){
                break;
            }

            if( empty($conditions->conditions) || ( $condition_remplie == true && $conditions->visible == true) || ( $condition_remplie == false && $conditions->visible != true) || !isset($condition_remplie) ){

                $corps_html .= '<span style="' . $this->config['onglet'] . '"><br>' . $onglet["libelle"] . ' </span><br />';
                $corps_html .= '<br />';
                $corps_html .= '<table cellspacing="4" cellpadding="3" >';
                if (is_array($onglet["lignes"]) && count($onglet["lignes"]) > 0) {
                    foreach ($onglet["lignes"] as $ligne) {
                        if ($ligne["type"] === "ligne_double_champ") {

                            // Vérification des conditions des champs 
                            $conditions = json_decode( base64_decode($ligne["champs"][0]['conditions'] ) ) ; 
                            $checkCondition = $this->checkCondition($conditions,$typeProfil);
                            if( !empty($conditions->conditions) ){
                                foreach ($conditions->conditions as $key => $condition) {
                                    $condition_remplie = false ;
                                    $valeur_champ = $fieldFactory->getValueChamp($condition->champ, $liste_champs[$condition->champ]['description']) ;
                                    $condition_remplie =  $this->is_condition(  $condition->value, $condition->operator ,$valeur_champ ,  $liste_champs[$condition->champ] );
                                    if( $conditions->verified == "all" && $condition_remplie == false ){ // Si toutes les conditions doivent être remplies 
                                        break;
                                    }
                                    elseif( $conditions->verified != "all" && $condition_remplie == true ){ // Si une des conditions est remplie on continue
                                        break;
                                    }
                                }
                            }
                            $champs_conditions_1 = false;
                            if( empty($conditions->conditions) || ( $condition_remplie == true && $conditions->visible == true) || ( $condition_remplie == false && $conditions->visible != true) || !isset($condition_remplie) ){
                                $champs_conditions_1 = true;
                            }
                            if((!empty($typeProfil) && !empty($conditions->type_profil) && empty($checkCondition))) $champs_conditions_1 = false;

                            
                            $conditions = json_decode( base64_decode($ligne["champs"][1]['conditions'] ) ) ; 
                            $checkCondition = $this->checkCondition($conditions,$typeProfil);
                            if( !empty($conditions->conditions) ){
                                foreach ($conditions->conditions as $key => $condition) {
                                    $condition_remplie = false ;
                                    $valeur_champ = $fieldFactory->getValueChamp($condition->champ, $liste_champs[$condition->champ]['description']) ;
                                    $condition_remplie =  $this->is_condition(  $condition->value, $condition->operator ,$valeur_champ ,  $liste_champs[$condition->champ] );
                                    if( $conditions->verified == "all" && $condition_remplie == false ){ // Si toutes les conditions doivent être remplies 
                                        break;
                                    }
                                    elseif( $conditions->verified != "all" && $condition_remplie == true ){ // Si une des conditions est remplie on continue
                                        break;
                                    }
                                }
                            }
                            $champs_conditions_2 = false;
                            if( empty($conditions->conditions) || ( $condition_remplie == true && $conditions->visible == true) || ( $condition_remplie == false && $conditions->visible != true) || !isset($condition_remplie) || (!empty($typeProfil) && !empty($conditions->type_profil) && !empty($checkCondition)) ){
                                $champs_conditions_2 = true;
                            }
                            if((!empty($typeProfil) && !empty($conditions->type_profil) && empty($checkCondition))) $champs_conditions_2 = false;

                            if( true == $champs_conditions_1 || true == $champs_conditions_2  ){ // Si les deux champs ne sont pas conditionnés et false

                                if ($this->config['in_colonnes'] == "1" ) {
                                    

                                    $corps_html .= '<tr>';
                                    if(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'wysiwyg' && $champs_conditions_1 ){
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                        $corps_html .= '<td COLSPAN="3" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'wysiwyg' && $champs_conditions_2){
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][1]["libelle"] . ' : </span></td>';
                                            $corps_html .= '<td COLSPAN="3" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][1]) . '</span></td>';
                                    }elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'note'  && $champs_conditions_1 ){
                                            $corps_html .= '<td COLSPAN="4" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'note' && $champs_conditions_2){
                                            $corps_html .= '<td COLSPAN="4" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][1]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'tableau'  && $champs_conditions_1 ){
                                            $corps_html .= '<td COLSPAN="4"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'tableau'  && $champs_conditions_2 ){
                                            $corps_html .= '<td COLSPAN="4"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][1]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'api_particulier' && $champs_conditions_1 ){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][0]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_numero_fiscal']) ? $value['label_numero_fiscal'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['numero_fiscal']) ? $value['numero_fiscal'] : '') . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_referece_fiscal']) ? $value['label_referece_fiscal'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['referece_fiscal']) ? $value['referece_fiscal'] : '') . '</span></td>';
                                        $corps_html .= '</tr>';
                                        $nombre = 1;

                                        if(isset($value['label_avis_impot']))
                                        {
                                            foreach($value['label_avis_impot'] as $nom => $label){
                                                if($nombre % 2 != 0){
                                                    $corps_html .= '<tr>';
                                                }
                                                $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                                $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['avis_impot'][$nom] . '</span></td>';
                                                if($nombre % 2 == 0 && count($value['label_avis_impot']) != $nombre){
                                                    $corps_html .= '</tr>';
                                                }
                                                $nombre++;
                                            }
                                            
                                        }
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'api_particulier' && $champs_conditions_2){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][1]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $value['label_numero_fiscal'] . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['numero_fiscal'] . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $value['label_referece_fiscal'] . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['referece_fiscal'] . '</span></td>';
                                        $corps_html .= '</tr>';
                                        $nombre = 1;
                                        foreach($value['label_avis_impot'] as $nom => $label){
                                            if($nombre % 2 != 0){
                                                $corps_html .= '<tr>';
                                            }
                                            $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                            $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['avis_impot'][$nom] . '</span></td>';
                                            if($nombre % 2 == 0 && count($value['label_avis_impot']) != $nombre){
                                                $corps_html .= '</tr>';
                                            }
                                            $nombre++;
                                        }
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'api_r2p' && $champs_conditions_1 ){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][0]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_nom']) ? $value['label_nom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_nom']) ? $value['value_api_r2p_nom'] : '') . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_prenom']) ? $value['label_prenom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_prenom']) ? $value['value_api_r2p_prenom'] : '') . '</span></td>';
                                        $corps_html .= '</tr>';

                                        $corps_html .= '<tr>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_date_naissance']) ? $value['label_date_naissance'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_date_naissance']) ? $value['value_api_r2p_date_naissance'] : '') . '</span></td>';
                                        $nombre = 0;

                                        if(isset($value['label_api_r2p']))
                                        {
                                            foreach($value['label_api_r2p'] as $nom => $label){
                                                if($nombre % 2 != 0){
                                                    $corps_html .= '<tr>';
                                                }
                                                $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                                $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['resultat'][$nom] . '</span></td>';
                                                if($nombre % 2 == 0 && count($value['label_api_r2p']) != $nombre){
                                                    $corps_html .= '</tr>';
                                                }
                                                $nombre++;
                                            }
                                        }
                                        $corps_html .= '</table><br><table cellspacing="4" cellpadding="3">';
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'api_r2p' && $champs_conditions_1 ){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][1]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_nom']) ? $value['label_nom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_nom']) ? $value['value_api_r2p_nom'] : '') . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_prenom']) ? $value['label_prenom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_prenom']) ? $value['value_api_r2p_prenom'] : '') . '</span></td>';
                                        $corps_html .= '</tr>';

                                        $corps_html .= '<tr>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_date_naissance']) ? $value['label_date_naissance'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_date_naissance']) ? $value['value_api_r2p_date_naissance'] : '') . '</span></td>';
                                        $nombre = 0;

                                        if(isset($value['label_api_r2p']))
                                        {
                                            foreach($value['label_api_r2p'] as $nom => $label){
                                                if($nombre % 2 != 0){
                                                    $corps_html .= '<tr>';
                                                }
                                                $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                                $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['resultat'][$nom] . '</span></td>';
                                                if($nombre % 2 == 0 && count($value['label_api_r2p']) != $nombre){
                                                    $corps_html .= '</tr>';
                                                }
                                                $nombre++;
                                            }
                                        }
                                        $corps_html .= '</table><br><table cellspacing="4" cellpadding="3">';
                                    }
                                    else{
                                        if (is_array($ligne["champs"][0]) && count($ligne["champs"][0]) > 0  && $champs_conditions_1) {
                                            $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                            $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                        }
                                        if (is_array($ligne["champs"][1]) && count($ligne["champs"][1]) > 0 && $champs_conditions_2) {
                                            if(count($ligne["champs"][0]) == 0){
                                                $corps_html .= '<td><span style="' . $this->config['label'] . '"></span></td>';
                                                $corps_html .= '<td><span style="' . $this->config['champ'] . '"></span></td>';
                                            }
                                            $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][1]["libelle"] . ' : </span></td>';
                                            $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][1]) . '</span></td>';
                                            
                                        }else{
                                            $corps_html .=      '<td><span style="' . $this->config['label'] . '"></span></td>';
                                            $corps_html .=      '<td><span style="' . $this->config['champ'] . '"></span></td>';
                                        }
                                    }

                                    $corps_html .= '</tr>';
                                } else {
                                    if(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'wysiwyg' && $champs_conditions_1 ){
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                            $corps_html .= '<td COLSPAN="3" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'wysiwyg' && $champs_conditions_2){
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                            $corps_html .= '<td COLSPAN="3" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'note'  && $champs_conditions_1 ){
                                            $corps_html .= '<td COLSPAN="4" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'note' && $champs_conditions_2){
                                            $corps_html .= '<td COLSPAN="4" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'tableau'  && $champs_conditions_1 ){
                                            $corps_html .= '<td COLSPAN="4"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'tableau' && $champs_conditions_2){
                                            $corps_html .= '<td COLSPAN="4"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'api_particulier' && $champs_conditions_1 ){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][0]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $value['label_numero_fiscal'] . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['numero_fiscal'] . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $value['label_referece_fiscal'] . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['referece_fiscal'] . '</span></td>';
                                        $corps_html .= '</tr>';
                                        $nombre = 1;
                                        foreach($value['label_avis_impot'] as $nom => $label){
                                            if($nombre % 2 != 0){
                                                $corps_html .= '<tr>';
                                            }
                                            $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                            $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['avis_impot'][$nom] . '</span></td>';
                                            if($nombre % 2 == 0 && count($value['label_avis_impot']) != $nombre){
                                                $corps_html .= '</tr>';
                                            }
                                            $nombre++;
                                        }
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'api_particulier' && $champs_conditions_2){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][1]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $value['label_numero_fiscal'] . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['numero_fiscal'] . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $value['label_referece_fiscal'] . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['referece_fiscal'] . '</span></td>';
                                        $corps_html .= '</tr>';
                                        $nombre = 1;
                                        foreach($value['label_avis_impot'] as $nom => $label){
                                            if($nombre % 2 != 0){
                                                $corps_html .= '<tr>';
                                            }
                                            $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                            $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['avis_impot'][$nom] . '</span></td>';
                                            if($nombre % 2 == 0 && count($value['label_avis_impot']) != $nombre){
                                                $corps_html .= '</tr>';
                                            }
                                            $nombre++;
                                        }
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'api_r2p' && $champs_conditions_1 ){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][0]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_nom']) ? $value['label_nom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_nom']) ? $value['value_api_r2p_nom'] : '') . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_prenom']) ? $value['label_prenom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_prenom']) ? $value['value_api_r2p_prenom'] : '') . '</span></td>';
                                        $corps_html .= '</tr>';

                                        $corps_html .= '<tr>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_date_naissance']) ? $value['label_date_naissance'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_date_naissance']) ? $value['value_api_r2p_date_naissance'] : '') . '</span></td>';
                                        $nombre = 0;

                                        if(isset($value['label_api_r2p']))
                                        {
                                            foreach($value['label_api_r2p'] as $nom => $label){
                                                if($nombre % 2 != 0){
                                                    $corps_html .= '<tr>';
                                                }
                                                $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                                $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['resultat'][$nom] . '</span></td>';
                                                if($nombre % 2 == 0 && count($value['label_api_r2p']) != $nombre){
                                                    $corps_html .= '</tr>';
                                                }
                                                $nombre++;
                                            }
                                        }
                                        $corps_html .= '</table><br><table cellspacing="4" cellpadding="3">';
                                    }
                                    elseif(is_array($ligne["champs"][1]) && $ligne["champs"][1]['type'] == 'api_r2p' && $champs_conditions_1 ){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][1]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_nom']) ? $value['label_nom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_nom']) ? $value['value_api_r2p_nom'] : '') . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_prenom']) ? $value['label_prenom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_prenom']) ? $value['value_api_r2p_prenom'] : '') . '</span></td>';
                                        $corps_html .= '</tr>';

                                        $corps_html .= '<tr>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_date_naissance']) ? $value['label_date_naissance'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_date_naissance']) ? $value['value_api_r2p_date_naissance'] : '') . '</span></td>';
                                        $nombre = 0;

                                        if(isset($value['label_api_r2p']))
                                        {
                                            foreach($value['label_api_r2p'] as $nom => $label){
                                                if($nombre % 2 != 0){
                                                    $corps_html .= '<tr>';
                                                }
                                                $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                                $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['resultat'][$nom] . '</span></td>';
                                                if($nombre % 2 == 0 && count($value['label_api_r2p']) != $nombre){
                                                    $corps_html .= '</tr>';
                                                }
                                                $nombre++;
                                            }
                                        }
                                        $corps_html .= '</table><br><table cellspacing="4" cellpadding="3">';
                                    }else{
                                        if (is_array($ligne["champs"][0]) && count($ligne["champs"][0]) > 0  && $champs_conditions_1 ) {
                                            $corps_html .= '<tr>';
                                            $corps_html .=      '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                            $corps_html .=      '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                            $corps_html .= '</tr>';
                                        }
                                        if (is_array($ligne["champs"][1]) && count($ligne["champs"][1]) > 0 && $champs_conditions_2) {
                                            $corps_html .= '<tr>';
                                            $corps_html .=      '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][1]["libelle"] . ' : </span></td>';
                                            $corps_html .=      '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][1]) . '</span></td>';
                                            $corps_html .= '</tr>';
                                        }else{
                                            $corps_html .=      '<td><span style="' . $this->config['label'] . '"></span></td>';
                                            $corps_html .=      '<td><span style="' . $this->config['champ'] . '"></span></td>';
                                        }
                                    }

                                }
                            }
                        } else if ($ligne["type"] === "section") {
                            // Vérification des conditions des champs 
                            $conditions = json_decode( base64_decode($ligne['conditions'] ) ) ; 
                            $checkCondition = $this->checkCondition($conditions,$typeProfil);
                            if( !empty($conditions->conditions)){
                                foreach ($conditions->conditions as $key => $condition) {
                                    $condition_remplie = false ;
                                    $valeur_champ = $fieldFactory->getValueChamp($condition->champ, $liste_champs[$condition->champ]['description']) ;
                                    $condition_remplie =  $this->is_condition(  $condition->value, $condition->operator ,$valeur_champ ,  $liste_champs[$condition->champ] );
                                    if( $conditions->verified == "all" && $condition_remplie == false ){ // Si toutes les conditions doivent être remplies 
                                        break;
                                    }
                                    elseif( $conditions->verified != "all" && $condition_remplie == true ){ // Si une des conditions est remplie on continue
                                        break;
                                    }
                                }
                            }
                            $champs_conditions_1 = false;
                            if( empty($conditions->conditions) || ( $condition_remplie == true && $conditions->visible == true) || ( $condition_remplie == false && $conditions->visible != true) || !isset($condition_remplie) ){
                                $champs_conditions_1 = true;
                            }
                            if((!empty($typeProfil) && !empty($conditions->type_profil) && empty($checkCondition))) $champs_conditions_1 = false;

                            if($champs_conditions_1 == true){
                                $corps_html .= '</table><br><table cellspacing="4" cellpadding="3">';
                                $corps_html .= '<tr>';
                                $corps_html .= '<td colspan="5" style="' . $this->config["section"] . 'border-bottom: 1pt solid black;" ><span>'.$ligne['title'].'</span></td>';
                                $corps_html .= '</tr></table><br><br><br><table cellspacing="4" cellpadding="3">';
                            }

                        } else if($ligne['type'] === 'ligne_simple_champ')
                        {
                            // Vérification des conditions des champs 
                            $conditions = json_decode( base64_decode($ligne["champs"][0]['conditions'] ) ) ; 
                            $checkCondition = $this->checkCondition($conditions,$typeProfil);
                            if( !empty($conditions->conditions)){
                                foreach ($conditions->conditions as $key => $condition) {
                                    $condition_remplie = false ;
                                    $valeur_champ = $fieldFactory->getValueChamp($condition->champ, $liste_champs[$condition->champ]['description']) ;
                                    $condition_remplie =  $this->is_condition(  $condition->value, $condition->operator ,$valeur_champ ,  $liste_champs[$condition->champ] );
                                    if( $conditions->verified == "all" && $condition_remplie == false ){ // Si toutes les conditions doivent être remplies 
                                        break;
                                    }
                                    elseif( $conditions->verified != "all" && $condition_remplie == true ){ // Si une des conditions est remplie on continue
                                        break;
                                    }
                                }
                            }
                            $champs_conditions_1 = false;
                            if( empty($conditions->conditions) || ( $condition_remplie == true && $conditions->visible == true) || ( $condition_remplie == false && $conditions->visible != true) || !isset($condition_remplie) ){
                                $champs_conditions_1 = true;
                            }
                            if((!empty($typeProfil) && !empty($conditions->type_profil) && empty($checkCondition))) $champs_conditions_2 = false;

                            if( true == $champs_conditions_1)
                            { // Si les deux champs ne sont pas conditionnés et false

                                if ($this->config['in_colonnes'] == "1" ) 
                                {
                                    $corps_html .= '<tr>';
                                    if(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'wysiwyg' && $champs_conditions_1 )
                                    {
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                        $corps_html .= '<td COLSPAN="3" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'note'  && $champs_conditions_1 )
                                    {
                                            $corps_html .= '<td COLSPAN="4" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'tableau'  && $champs_conditions_1 )
                                    {
                                            $corps_html .= '<td COLSPAN="4"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'api_particulier' && $champs_conditions_1 )
                                    {
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][0]);

                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_numero_fiscal']) ? $value['label_numero_fiscal'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['numero_fiscal']) ? $value['numero_fiscal'] : '') . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_referece_fiscal']) ? $value['label_referece_fiscal'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['referece_fiscal']) ? $value['referece_fiscal'] : '') . '</span></td>';
                                        $corps_html .= '</tr>';
                                        $nombre = 1;

                                        if(isset($value['label_avis_impot']))
                                        {
                                            foreach($value['label_avis_impot'] as $nom => $label){
                                                if($nombre % 2 != 0){
                                                    $corps_html .= '<tr>';
                                                }
                                                $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                                $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['avis_impot'][$nom] . '</span></td>';
                                                if($nombre % 2 == 0 && count($value['label_avis_impot']) != $nombre){
                                                    $corps_html .= '</tr>';
                                                }
                                                $nombre++;
                                            }
                                            
                                        }
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'api_r2p' && $champs_conditions_1 ){
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][0]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_nom']) ? $value['label_nom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_nom']) ? $value['value_api_r2p_nom'] : '') . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_prenom']) ? $value['label_prenom'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_prenom']) ? $value['value_api_r2p_prenom'] : '') . '</span></td>';
                                        $corps_html .= '</tr>';

                                        $corps_html .= '<tr>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . (isset($value['label_date_naissance']) ? $value['label_date_naissance'] : '') . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . (isset($value['value_api_r2p_date_naissance']) ? $value['value_api_r2p_date_naissance'] : '') . '</span></td>';
                                        $nombre = 0;

                                        if(isset($value['label_api_r2p']))
                                        {
                                            foreach($value['label_api_r2p'] as $nom => $label){
                                                if($nombre % 2 != 0){
                                                    $corps_html .= '<tr>';
                                                }
                                                $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                                $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['resultat'][$nom] . '</span></td>';
                                                if($nombre % 2 == 0 && count($value['label_api_r2p']) != $nombre){
                                                    $corps_html .= '</tr>';
                                                }
                                                $nombre++;
                                            }
                                        }
                                        $corps_html .= '</table><br><table cellspacing="4" cellpadding="3">';
                                    }
                                    else
                                    {
                                        if (is_array($ligne["champs"][0]) && count($ligne["champs"][0]) > 0  && $champs_conditions_1) {
                                            $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                            $corps_html .= '<td COLSPAN="3" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                        }
                                    }

                                    $corps_html .= '</tr>';
                                } 
                                else 
                                {
                                    if(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'wysiwyg' && $champs_conditions_1 )
                                    {
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                            $corps_html .= '<td COLSPAN="3" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'note'  && $champs_conditions_1 )
                                    {
                                            $corps_html .= '<td COLSPAN="4" bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                    }
                                    elseif(is_array($ligne["champs"][0]) && $ligne["champs"][0]['type'] == 'api_particulier' && $champs_conditions_1 )
                                    {
                                        $value = $fieldFactory->getChampPdfValue($ligne["champs"][0]);
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $value['label_numero_fiscal'] . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['numero_fiscal'] . '</span></td>';
                                        $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $value['label_referece_fiscal'] . ' : </span></td>';
                                        $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['referece_fiscal'] . '</span></td>';
                                        $corps_html .= '</tr>';
                                        $nombre = 1;
                                        foreach($value['label_avis_impot'] as $nom => $label)
                                        {
                                            if($nombre % 2 != 0)
                                            {
                                                $corps_html .= '<tr>';
                                            }
                                            $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $label . ' : </span></td>';
                                            $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $value['avis_impot'][$nom] . '</span></td>';
                                            if($nombre % 2 == 0 && count($value['label_avis_impot']) != $nombre)
                                            {
                                                $corps_html .= '</tr>';
                                            }
                                            $nombre++;
                                        }
                                    }
                                    else
                                    {
                                        if (is_array($ligne["champs"][0]) && count($ligne["champs"][0]) > 0  && $champs_conditions_1 ) 
                                        {
                                            $corps_html .= '<tr>';
                                            $corps_html .=      '<td><span style="' . $this->config['label'] . '">' . $ligne["champs"][0]["libelle"] . ' : </span></td>';
                                            $corps_html .=      '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . $fieldFactory->getChampPdfValue($ligne["champs"][0]) . '</span></td>';
                                            $corps_html .= '</tr>';
                                        }
                                    }

                                }
                            }
                        }
                    }
                }
                $corps_html .= '</table>';
            }
        }

        $this->writeHTML($corps_html, true, false, true, false, '');
    }


    /****************************************************************************************************************************************/
    /*********************************************************** Surcharge TCPDF ***********************************************************/
    /****************************************************************************************************************************************/

    /**
     * @access public
     * @name Header()
     * Fonction qui construit le header du pdf  ( Surcharge fonction TCPDF::Header )
     *
     */
    public function Header()
    {

        $table =    '<table>
                        <tr>
                            <td colspan="2"><img height="60px" width="auto" src="' . $this->config["logo_chemin"] . '" align="left"></td>
                            <td ></td>
                            <td ></td>
                            <td ></td>
                            <td colspan="2"><span style="font-size:12px;text-align:right;">' . $this->header["dossier_date"] . ' </span> </td>
                        </tr>
                        <tr>
                            <td ></td>
                            <td colspan="5" style="text-align: center;padding-bottom:10px;"> 
                                <span style="' . $this->config["titre"] . '" >' .  $this->header["dispositif_name"] . '</span> <br />
                                <span style="' . $this->config["titre"] . '" > 
                                    Dossier N° ' . $this->header["dossier_num"] . '<br />
                                </span> 
                                <span style="' . $this->config["titre"] . '" >' .  $this->header["demandeur_name"] . '</span> 
                            </td>
                            <td ></td>
                        </tr>
                    </table>';

        //Ecriture du tableau dans l'êntete
        $this->writeHTML($table, true, false, true, false, '');
    }

    /**
     * @access public
     * @name Footer()
     * Fonction qui construit le footer du pdf  ( Surcharge fonction TCPDF::Footer )
     *
     */
    public function Footer()
    {
        // Position at 15 mm from bottom
        $this->SetY(-15);
        // Set font
        $this->SetFont('helvetica', 'I', 8);
        // Page number
        $this->Cell(0, 10, 'Page ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, false, 'C', 0, '', 0, false, 'T', 'M');
    }

    private function is_condition( $condition_value , $operateur , $champ_value, $champ_type ){

        // Retrait des simples coat sinon => erreur 
        str_replace("'", "", $condition_value );
        str_replace("'", "", $champ_value );

        $operateur = ( $operateur == "=" ) ? "==" : $operateur;
        switch ($champ_type['type']) {
            case 'checkbox' :
                $condition_value = ( !empty($condition_value) && $condition_value != 'false' && $condition_value != '0' ) ? '1' : '0'; 
                $eval_condition = "'".$champ_value."'=='".$condition_value."'";
                $retour = eval('return ('.$eval_condition.');') ; 
                break;
            case 'checkboxmulti' :
                $champ_value = str_replace("^", "", $champ_value );
                $array_value = explode(",", $champ_value);
                $retour = in_array($condition_value, $array_value);
                break;
            case 'liste' :
                $params = json_decode( base64_decode($champ_type['params']) ) ; 
                if( $params->multiple == true ){
                    $champ_value = str_replace("^", "", $champ_value );
                    $array_value = explode(",", $champ_value);
                    $retour = in_array($condition_value, $array_value) ; 
                }
                else{
                    $champ_value = str_replace("^", "", $champ_value );
                    $eval_condition = "'".$champ_value."'".$operateur."'".$condition_value."'";
                    $retour = eval('return ('.$eval_condition.');') ; 
                }
                break;
            case 'nombre' :
                $champ_value = (empty($champ_value))?0:$champ_value;
                $champ_value = number_format($champ_value, 2);
                $condition_value = number_format($condition_value, 2);
                $eval_condition = "'".$champ_value."'".$operateur."'".$condition_value."'";
                $retour = eval('return ('.$eval_condition.');') ; 
                break;
            case 'date' :
                // Vérification si la date de condition est saisie en anglais "2000-12-30" ou fr "30/12/2000"
                if( empty( $champ_value ) || strlen($champ_value) != 10 || strlen($condition_value) != 10 ){
                    return false ;
                }

                if( strpos($condition_value, "-") === false ){
                    $date_condition = DateTime::createFromFormat('d/m/Y', $condition_value);
                    $date_condition = $date_condition->format('Y-m-d');
                }
                else{
                    $date_condition = $condition_value ; 
                }            
                $date_value = DateTime::createFromFormat('d/m/Y', $champ_value);
                $date_value = $date_value->format('Y-m-d');

                $eval_condition = "'".$date_value."'".$operateur."'".$date_condition."'";
                $retour = eval('return ('.$eval_condition.');') ; 
                break;
            case 'api_particulier':
                $params = json_decode( base64_decode($champ_type['params']),true);
                $value = str_replace('&amp;#x3D;','=',$champ_value);
                $value = json_decode( base64_decode($value),true);
                $value_condition = json_decode( base64_decode($value["avis_impot"]),true);
                $eval_condition = "'".$value_condition[$params['avis_impot_condition']]."'".$operateur."'".$condition_value."'";
                $retour = eval('return ('.$eval_condition.');') ; 
                break;
            default:
                $eval_condition = "'".$champ_value."'".$operateur."'".$condition_value."'";
                $retour = eval('return ('.$eval_condition.');') ; 
                break;
        }

        return $retour ;
    }

    private function checkCondition($conditions,$typeProfil){
        if(!empty($typeProfil) && !empty($conditions->type_profil) && is_array($conditions->type_profil) && strpos(json_encode($conditions->type_profil), $typeProfil) !== false && strpos(json_encode($conditions->type_profil), $typeProfil) >= 0){
            return true;
        }else{
            return false;
        }
    }
}
